#!/usr/bin/perl -w
#
# Copyright (c) 1999 - 2011 Schneider Electric, All Rights Reserved.
#
# This scripts allows users to shutdown/reboot host. 
# The supported operations are: reboot and shutdown. 
# For more information, execute the script with --help option.
#
# Author: tyamazoe
# Version: $Id: shutdownhost.pl 2079 2011-02-17 11:47:04Z ccollins $
#

use strict;
use warnings;

use VMware::VIRuntime;
use VMware::VIFPLib;


$SIG{__DIE__} = sub{Util::disconnect()};
$Util::script_version = "1.0";

sub shutdown_operation;

my $level = 0;

my %operations = (
   reboot => \&reboot,
   shutdown => \&shutdown,
);

my %opts = (
   operation => {
      type => "=s",
      help => "Operation to perform on target host: " . 
          join(", ", sort keys %operations),
      required => 1,
   },
   logoutput => {
        type => "=s",
        required => 0,
        },
   loglevel => {
        type => "=s",
        required => 0,
        },
);


# read/validate options
Opts::add_options(%opts);
Opts::parse();
Opts::validate();

my $operation = Opts::get_option('operation');
my $log_output = Opts::get_option('logoutput');
my $loglevel = Opts::get_option('loglevel');

if ($loglevel eq "debug") {
                $log_output = "/tmp/PCNSHostShutdown.log";
                open STDOUT, ">>", $log_output;
                open STDERR, ">>", $log_output;
}
else
{
                $log_output = "&STDOUT";
}
        
# validation: verify that operation is supported
if (!exists($operations{$operation})) {
    print "Unknown operation: '$operation'\n";
    &log("debug", "Unknown operation: '$operation'\n");
    print "List of valid operations:";
    map { print "   $_\n"; } sort keys %operations;
    Opts::usage();
    exit 1;
}

my $server = Opts::get_option('server');
if ($server) {
    Util::trace(0, "\nConnecting to server $server ...\n");
    &log("debug", "Connecting to server $server\n");

}
Util::trace(0, "Operation :$operation\n");

# connect to the server
##Util::connect();

# get list of fastpass targets
my @targets = VIFPLib::enumerate_targets();
foreach my $target(@targets){
    # connect to the lib using fastpass
    eval {
    Util::trace(0, "Version on $target: \n");
    Util::trace(0, "-------------------------------------------------------\n");
    &log("debug", "Login by vi-fastpass to: " . $target);

    VIFPLib::login_by_fastpass($target);

    # shutdown host system
    shutdown_operation();

    # disconnect from the server
    Util::disconnect();
    &log("debug", "Disconnected.\n");
    Util::trace(0, "Disconnected.\n");
    };
        if($@) {
        &log("debug","Error: Unable to login to host: ". "\"$target\"! Host may not be accessible over the network or is not managed by VMA");
        }
}

##

sub shutdown_operation() {
    my $name = 'host';
    my $folder_views = Vim::find_entity_views (view_type => 'Folder',
                                               filter => {name => $name});
    unless (@$folder_views) {
        Util::trace(0, "Folder $name not found.\n");
        &log("debug", "Folder $name not found.\n");
        return;
    }
    foreach(@$folder_views) {
        Util::trace(0, "\nFolder :$name \n");
        &log("debug", "\nFolder :$name \n");
        my $folder = $_;
        my $result = get_entities(view_type => 'HostSystem',
                                  obj => $folder);
        foreach (@$result) {
            my $obj_content = $_;
            my $mob = $obj_content->obj;
            my $obj = Vim::get_view(mo_ref=>$mob);
            if ($obj) {
                Util::trace(0, "Host :".$obj->name." \n");
                if ($operation eq 'shutdown') {
                    Util::trace(0, "Shutting down host ".$obj->name);
                    &log("debug", "Shutting down host ".$obj->name."\n");
                    shutdown_host($obj);
                } elsif ($operation eq 'reboot') {
                    Util::trace(0, "Rebooting host ".$obj->name);
                    reboot_host($obj);
                } else {
                    Util::trace(0, "Unknown operation $operation.\n");
                    &log("debug", "Unknown operation $operation.\n");
                }
            } else {
                Util::trace(0, "Can't shutdown/reboot system.\n");
                &log("debug", "Can't shutdown/reboot system.\n");
            }
        }
    }
}

sub reboot_host {
    my $host = shift;
    $host->RebootHost(force => 1);
    Util::trace(0, "done\n\n");
    &log("debug", "\ndone\n\n");
}

sub shutdown_host {
    my $host = shift;
    $host->ShutdownHost(force => 1);
    Util::trace(0, "done\n\n");
    &log("debug", "\ndone\n\n");
}

sub get_entities {
    my %args = @_;
    my $view_type = $args{view_type};
    my $obj = $args{obj};
    
    my $sc = Vim::get_service_content();
    my $service = Vim::get_vim_service();

    my $property_spec = PropertySpec->new(all => 0,
                                          type => $view_type->get_backing_type());
    my $property_filter_spec = $view_type->get_search_filter_spec($obj,[$property_spec]);
    my $obj_contents = $service->RetrieveProperties(_this => $sc->propertyCollector,
                                                   specSet => $property_filter_spec);
    my $result = Util::check_fault($obj_contents);
    return $result;
}

sub log {
        my($logLevel, $message) = @_;

        open(LOG,">>$log_output");
        print LOG GetDateTimeFormatted(), $logLevel, ": ", $message,"\n";
        close(LOG);
}


sub GetDateTimeFormatted {

    my (@sDateTimehash) = ();
    my ($sPar)          = '';
    my ($sDateAndTime)  = '';
    if ( defined $_[0] != 0 ) {
        $sPar = $_[0];
    }

    @sDateTimehash    = localtime;
    $sDateTimehash[4] = $sDateTimehash[4] + 1;
    $sDateTimehash[5] = $sDateTimehash[5] - 100 + 2000;

    if ( length( $sDateTimehash[0] ) < 2 ) {
        $sDateTimehash[0] = '0' . $sDateTimehash[0];
    }
    if ( length( $sDateTimehash[1] ) < 2 ) {
        $sDateTimehash[1] = '0' . $sDateTimehash[1];
    }
    if ( length( $sDateTimehash[2] ) < 2 ) {
        $sDateTimehash[2] = '0' . $sDateTimehash[2];
    }
    if ( length( $sDateTimehash[3] ) < 2 ) {
        $sDateTimehash[3] = '0' . $sDateTimehash[3];
    }
    if ( length( $sDateTimehash[4] ) < 2 ) {
        $sDateTimehash[4] = '0' . $sDateTimehash[4];
    }

    $sDateAndTime =
"[$sDateTimehash[3]\.$sDateTimehash[4]\.$sDateTimehash[5]\, $sDateTimehash[2]\:$sDateTimehash[1]\:$sDateTimehash[0]]";

    if ( $sPar eq 'd' ) {
        $sDateAndTime =
          "$sDateTimehash[3]\.$sDateTimehash[4]\.$sDateTimehash[5]";
    }
    if ( $sPar eq 't' ) {
        $sDateAndTime = "$sDateTimehash[2]$sDateTimehash[1]$sDateTimehash[0]";
    }

    return $sDateAndTime;
}
### EOF ###
